#region References

using System;
using System.Xml;
using System.Data;
using System.Collections;
using System.Reflection;
using System.Text.RegularExpressions;

using gov.va.med.vbecs.DAL;
using gov.va.med.vbecs.Common;
using gov.va.med.vbecs.ExceptionManagement;

using COMPONENTCLASS = gov.va.med.vbecs.Common.VbecsTables.ComponentClass;
using ARTIFICIAL = gov.va.med.vbecs.Common.DatabaseConstants.ArtificialColumnNames;
using PRODUCTTYPE = gov.va.med.vbecs.Common.VbecsTables.ProductType;
using HCPCSASSIGNMENT = gov.va.med.vbecs.Common.VbecsTables.HcpcsAssignment;
using PRODUCTSHIPPER = gov.va.med.vbecs.Common.VbecsTables.ProductShipper;
using BLOODPRODUCT = gov.va.med.vbecs.Common.VbecsTables.BloodProduct;

#endregion

namespace gov.va.med.vbecs.BOL
{
	#region Header

	///<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
	///<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
	///<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
	///<Developers>
	///	<Developer>Cameron Taylor</Developer>
	///</Developers>
	///<SiteName>Hines OIFO</SiteName>
	///<CreationDate>9/3/2002</CreationDate>
	///<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
	///<summary>Blood Product business layer class.</summary>

	#endregion

	public class BloodProduct: BloodComponentClass
	{
		#region Variables

		private DataRow _drBloodProduct;

		/// <summary>
		/// DataTable of Hcpcs codes - Static to avoid multiple loads (performance tune)
		/// </summary>
		private static DataTable _dtHcpcsCodes = null;

		/// <summary>
		/// Product Type.
		/// </summary>
		private BOL.ProductType _productType;

		/// <summary>
		/// Blood product unique identifier.
		/// </summary>
		private System.Guid _bloodProductGUID;

		/// <summary>
		/// Standard 5 character blood product identifier. Codabar is 5 digits, ISBT128 is alphanumeric.
		/// </summary>
		private string _productCode;

		/// <summary>
		/// Blood product name. Short form used for display purposes.
		/// </summary>
		private string _productShortName;

		/// <summary>
		/// Blood product name. Long form, supplied nationally.
		/// </summary>
		private string _productName;

		/// <summary>
		/// Blood product attributes
		/// </summary>
		private string _productAttributeForm;

		/// <summary>
		/// Average amount of a unit of this blood product in mL.
		/// </summary>
		private System.Int16 _productVolume;

		/// <summary>
		/// The maximum number of days that this unit may remain in storage.
		/// </summary>
		private int _maximumStorageDays;

		/// <summary>
		/// The maximum number of hours (less whole days) that this unit may remain in storage.
		/// </summary>
		private int _maximumStorageHours;

		/// <summary>
		/// A code that is used in the billing process for a blood product package.
		/// </summary>
		private string _hcpcsCode;

		/// <summary>
		/// Collection of Shippers of this blood product.
		/// </summary>
		private BOL.ProductShipperCollection _productShippers;



		/// <summary>
		/// Blood product barcode type. (Codabar or ISBT128).
		/// </summary>
		private BarcodeType _barcodeType;

		private string _unitDivisionID = string.Empty;

		private DonationType _donationType;

		private bool _frozen;

		#endregion

		#region Events

		/// <summary>
		/// When a product Shipper is added/removed/updated for this bloodproduct
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		protected virtual void OnProductShippersChanged(object sender, System.EventArgs e)
		{
			RuleBroken("ProductShippers", (this._productShippers.Count == 0));
		}

		#endregion

		#region Constructors

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/11/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3801"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>Valid BloodProduct Object</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="569"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// BloodProduct class constructor.  Retrieves blank blood product object
		/// </summary>		
		public BloodProduct()
		{
			this.Clear();
		}

		///<Developers>
		///	<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/21/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3355"> 
		///		<ExpectedInput>Valid GUID</ExpectedInput>
		///		<ExpectedOutput>Valid Blood Product object</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3356"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Blood Product constructor
		/// </summary>
		/// <param name="bloodProductGuid"></param>
		public BloodProduct(System.Guid bloodProductGuid)
		{
			this.GetBloodProductByGuid(bloodProductGuid);
		}

		///<Developers>
		///	<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/21/2003</CreationDate>
		/// <summary>
		/// BloodProduct
		/// </summary>
		/// <param name="productCode"></param>
		/// <param name="bloodProductGuid"></param>
		protected BloodProduct(string productCode, System.Guid bloodProductGuid)
		{
			this.Clear();
			this.ProductCode = productCode;
			this.GetBloodProductByGuid(bloodProductGuid);
			this.ProductShippers = new BOL.ProductShipperCollection(System.Guid.NewGuid());
			this.ProductShippers.ListChanged += new System.EventHandler(OnProductShippersChanged);
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/11/2002</CreationDate>
		///
		///<TestCases>
		///	<Case type="0" testid ="2958"> 
		///		<ExpectedInput>Valid data row</ExpectedInput>
		///		<ExpectedOutput>Valid BloodProduct object</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3334"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// BloodProduct class constructor. Loads properties based on given blood product datarow.
		/// </summary>
		/// <param name="dataRow"></param>
		public BloodProduct(DataRow dataRow)
		{
			this.LoadFromDataRow(dataRow);
			this.IsDirty = false;
		}

		#endregion

		#region Properties

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/3/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="336"> 
		///		<ExpectedInput>Guid</ExpectedInput>
		///		<ExpectedOutput>blood product GUID</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="500"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets or sets blood product unique identifier.
		/// </summary>
		public System.Guid BloodProductGuid
		{
			get
			{
				return _bloodProductGUID;
			}
			set
			{
				this._bloodProductGUID = value;
				this.IsDirty = true;
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/3/2002</CreationDate>
		///<TestCases>
		/// 
		///<Case type="0" testid ="3346"> 
		///		<ExpectedInput>Valid string</ExpectedInput>
		///		<ExpectedOutput>Valid string</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="570"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// ProductAttributeForm
		/// </summary>
		public string ProductAttributeForm
		{
			get
			{
				return this._productAttributeForm;
			}
			set
			{
				this._productAttributeForm = value;
				this.IsDirty = true;
			}
		}

		private void DetermineIfFrozen()
		{
			switch(Common.Utility.GetProductTypeFromProductTypeCode(this.ProductType.ProductTypeCode))
			{
				case Common.ProductType.FrozenRedBloodCells:
					this._frozen = true;
					break;
				case Common.ProductType.FrozenRejuvenatedRedBloodCells:
					this._frozen = true;
					break;
				case Common.ProductType.FreshFrozenPlasma:
					this._frozen = true;
					break;
				case Common.ProductType.ApheresisFreshFrozenPlasma:
					this._frozen = true;
					break;
				case Common.ProductType.Plasma: //Added per CR3065  (BR_73.27 update in build 1.6.0.4)
					this._frozen = true;
					break;
				case Common.ProductType.FrozenApheresisPlatelets:
					this._frozen = true;
					break;
				case Common.ProductType.Cryoprecipitate:
					this._frozen = true;
					break;
				case Common.ProductType.PooledCryoprecipitate:
					this._frozen = true;
					break;
				case Common.ProductType.ApheresisCryoprecipitate:
					this._frozen = true;
					break;
				case Common.ProductType.FrozenApheresisRedBloodCells:
					this._frozen = true;
					break;
				case Common.ProductType.FrozenRejuvenatedApheresisRedBloodCells:
					this._frozen = true;
					break;
				case Common.ProductType.FrozenPooledSerum:
					this._frozen = true;
					break;
				default:
					this._frozen = false;
					break;
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/3/2002</CreationDate>
		///<TestCases>
		///
		///	<Case type="0" testid ="3342"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>Valid boolean</ExpectedOutput>
		///	</Case>
		///	
		///	<Case type="1" testid ="571"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///	
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// IsFrozen
		/// </summary>
		public bool IsFrozen
		{
			get
			{
				DetermineIfFrozen();
				return this._frozen;
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/3/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="337"> 
		///		<ExpectedInput>Valid product code</ExpectedInput>
		///		<ExpectedOutput>Valid product code</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="501"> 
		///		<ExpectedInput>Invalid product code</ExpectedInput>
		///		<ExpectedOutput>Business Object Exception</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="502"> 
		///		<ExpectedInput>Product code with an invalid donation type</ExpectedInput>
		///		<ExpectedOutput>Business Object Exception</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets or sets 5 character blood product identifier. Codabar is 5 digits, ISBT128 is alphanumeric.
		/// If Product code is ISBT-128, donation type will be defined by 6th character
		/// Implements BR_1.13
		/// </summary>
		public string ProductCode
		{
			get
			{
				return _productCode;
			}
			set
			{
				Regex _regexCodabar = Common.RegularExpressions.CodabarProductCode();
				Regex _regexISBT128 = Common.RegularExpressions.ISBT128ProductCode();
				//
				if ((value.Length > 0) && !_regexCodabar.IsMatch(value) && !_regexISBT128.IsMatch(value))
				{
					string errorMessage = Common.StrRes.SysErrMsg.Common.InvalidFormat(Common.Utility.ParseStringwithSpaces(MethodBase.GetCurrentMethod().Name, true)).ResString;
					throw new BusinessObjectException(string.Concat(errorMessage, ": ", value));
				}
				//
				if (_regexISBT128.IsMatch(value))
				{
					value = value.Replace("=<", "");
					this._barcodeType = Common.BarcodeType.ISBT128;
					if (value.Length > 5)
					{
						try
						{
							this.DonationType.DonationTypeCode = value[5];
						}
						catch (Common.DataAccessLayerException err)
						{
							throw new BOL.BusinessObjectException(err.Message);
						}
						if (value.Length == 8)
						{
							this.UnitDivisionId = value[6].ToString() + value[7].ToString();
						}
					}
				}
					//BR_1.13
				else
				{
					if (_regexCodabar.IsMatch(value))
					{
						this._barcodeType = Common.BarcodeType.Codabar;
					}
					else
					{
						if (value.Length > 0) //I think if block can be removed because it should never get hit since the
							//first if says if Length > 0 and !codabar && !isbt
						{
							string errorMessage = Common.StrRes.SysErrMsg.Common.InvalidPropertyOrMethod(
								Common.Utility.ParseStringwithSpaces(MethodBase.GetCurrentMethod().Name, true),
								value).ResString;
							throw new BusinessObjectException(errorMessage);
						}
						else //this to I think will never get hit, right?
						{
							this._barcodeType = Common.BarcodeType.None;
						}
					}
				}
				if (value.Length > 4)
				{
					value = value.Substring(0, 5);
				}
				_productCode = value;
				RuleBroken(MethodBase.GetCurrentMethod().Name, (value.Length == 0));
				this.IsDirty = true;
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/3/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="338"> 
		///		<ExpectedInput>Valid shortname</ExpectedInput>
		///		<ExpectedOutput>Valid shortname</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="503"> 
		///		<ExpectedInput>Invalid shortname Length > 40</ExpectedInput>
		///		<ExpectedOutput>Exception</ExpectedOutput>
		///	</Case>
		///	
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets and sets short form of product name used for display purposes.
		/// </summary>
		public string ProductShortName
		{
			get
			{
				return _productShortName;
			}
			set
			{
				if (value.Length > 50)
				{
					string errorMessage = Common.StrRes.SysErrMsg.Common.CannotbeGreaterError(value, "50").ResString;
					throw new BusinessObjectException(errorMessage);
				}
				_productShortName = value;
				RuleBroken(MethodBase.GetCurrentMethod().Name, (value.Length == 0));
				this.IsDirty = true;
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/3/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="339"> 
		///		<ExpectedInput>Valid product name</ExpectedInput>
		///		<ExpectedOutput>Valid product name</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="505"> 
		///		<ExpectedInput>Invalid product name length > 175</ExpectedInput>
		///		<ExpectedOutput>Business Object Exception</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets or sets long form of blood product name, supplied nationally
		/// </summary>
		public string ProductName
		{
			get
			{
				return _productName;
			}
			set
			{
				if (value.Length > 175)
				{
					string errorMessage = Common.StrRes.SysErrMsg.Common.CannotbeGreaterError(Common.Utility.ParseStringwithSpaces(
						MethodBase.GetCurrentMethod().Name, true), "175").ResString;
					throw new BusinessObjectException(errorMessage);
				}
				_productName = value;
				RuleBroken(MethodBase.GetCurrentMethod().Name, (value.Length == 0));
				this.IsDirty = true;
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/11/2002</CreationDate>
		///
		///<TestCases>
		///<Case type="0" testid ="2469"> 
		///		<ExpectedInput>Valid product type</ExpectedInput>
		///		<ExpectedOutput>Valid product type</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="572"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets or sets contained product type
		/// </summary>
		public BOL.ProductType ProductType
		{
			get
			{
				if (this._productType == null)
				{
					this._productType = new BOL.ProductType();
				}
				return this._productType;
			}
			set
			{
				this._productType = value;
				RuleBroken(MethodBase.GetCurrentMethod().Name, (value == null));
				this.IsDirty = true;
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/3/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="342"> 
		///		<ExpectedInput>Valid product volume</ExpectedInput>
		///		<ExpectedOutput>Valid product volume</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="674"> 
		///		<ExpectedInput>Invalid product volume value less than 0</ExpectedInput>
		///		<ExpectedOutput>Argument Exception</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="573"> 
		///		<ExpectedInput>Invalid product volume value greater than 2000</ExpectedInput>
		///		<ExpectedOutput>Argument Exception</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets or sets average volume of a unit of this blood product in mL.
		/// </summary>
		public System.Int16 ProductVolume
		{
			get
			{
				return _productVolume;
			}
			set
			{
				if (value < 0 || value > 2000)
				{
					string errorMessage = Common.StrRes.SysErrMsg.Common.CannotbeLessError(value, "0").ResString;
					throw new ArgumentException(errorMessage);
				}
				_productVolume = value;
				RuleBroken(MethodBase.GetCurrentMethod().Name, (value < 0));
				this.IsDirty = true;
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/3/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="343"> 
		///		<ExpectedInput>Valid number of storage days</ExpectedInput>
		///		<ExpectedOutput>Valid number of storage days</ExpectedOutput>
		///</Case>
		///
		///<Case type="1" testid ="676"> 
		///		<ExpectedInput>Invalid number of storage days</ExpectedInput>
		///		<ExpectedOutput>Exception</ExpectedOutput>
		///</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets or sets the maximum number of days that this unit may remain in storage.
		/// </summary>
		public int MaximumStorageDays
		{
			get
			{
				return _maximumStorageDays;
			}
			set
			{
				if (value < 0)
				{
					string errorMessage = Common.StrRes.SysErrMsg.Common.CannotbeLessError(value, "0").ResString;
					throw new ArgumentException(errorMessage);
				}
				_maximumStorageDays = value;
				this.IsDirty = true;
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/3/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="344"> 
		///		<ExpectedInput>Valid number of storage hours</ExpectedInput>
		///		<ExpectedOutput>Valid number of storage hours</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="678"> 
		///		<ExpectedInput>Invalid number of storage hours less than 0</ExpectedInput>
		///		<ExpectedOutput>Business Object Exception</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="682"> 
		///		<ExpectedInput>Invalid number of storage hours greater than 23</ExpectedInput>
		///		<ExpectedOutput>Business Object Exception</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets or sets the maximum number of hours (less whole days) that this unit may remain in storage.
		/// </summary>
		public int MaximumStorageHours
		{
			get
			{
				return _maximumStorageHours;
			}
			set
			{
				if ((value < 0) || (value > 23))
				{
					string errorMessage = Common.StrRes.SysErrMsg.Common.RangeError(value, "0", "23").ResString;
					throw new BOL.BusinessObjectException(errorMessage);
				}
				_maximumStorageHours = value;
				this.IsDirty = true;
			}
		}

		///<Developers>
		///	<Developer>Greg</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/24/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6487"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6488"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Returns TRUE if the BloodProduct is found ACTIVE units for the current division
		/// and Shipped by the LocalSupplier
		/// </summary>
		/// <param name="bloodProductGuid"></param>
		/// <param name="localSupplierGuid"></param>
		/// <returns></returns>
		public static bool IsBloodProductOnActiveUnits(System.Guid bloodProductGuid, System.Guid localSupplierGuid)
		{
			return (DAL.BloodProduct.IsBloodProductOnActiveUnits(bloodProductGuid, localSupplierGuid));
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/3/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="345"> 
		///		<ExpectedInput>boolen</ExpectedInput>
		///		<ExpectedOutput>Product Active Indicator</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="679"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Indicates whether the blood product is active for the user's division.
		/// </summary>
		public bool ProductActiveIndicator
		{
			get
			{
				return DAL.BloodProduct.IsBloodProductActive(this.BloodProductGuid);
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/3/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="346"> 
		///		<ExpectedInput>Valid barcode type</ExpectedInput>
		///		<ExpectedOutput>Barcode type</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="680"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets or sets the blood product barcode type. (Codabar or ISBT128).
		/// </summary>
		public BarcodeType BarcodeType
		{
			get
			{
				return this._barcodeType;
			}
			set
			{
				this._barcodeType = value;
				this.IsDirty = true;
			}
		}

		///<Developers>
		///	<Developer>Mike      </Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/14/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2591"> 
		///		<ExpectedInput>Valid string</ExpectedInput>
		///		<ExpectedOutput>Valid HcpcsCode</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2592"> 
		///		<ExpectedInput>Invalid Hcpcs Code length > 5</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets or sets 5 character HCPCS code. 
		/// </summary>
		public string HcpcsCode
		{
			get
			{
				return this._hcpcsCode;
			}
			set
			{
				if (value.Length > 5)
				{
					string errorMessage = Common.StrRes.SysErrMsg.Common.FieldLengthError(
						Common.Utility.ParseStringwithSpaces(MethodBase.GetCurrentMethod().Name, true),
						"1", "5").ResString;
					throw new ArgumentException(errorMessage);
				}
				RuleBroken(MethodBase.GetCurrentMethod().Name, (value.Trim().Length == 0));
				this._hcpcsCode = value;
				this.IsDirty = true;
			}
		}
 
		///<Developers>
		///	<Developer></Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/8/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7376"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7377"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Returns if the BloodProduct has a B2 attribute, which means the unit is OPEN(contaminated)
		/// </summary>
		public bool HasB2Attribute
		{
			get
			{
				if (this.ProductAttributeForm == string.Empty)
				{
					DataTable dt = DAL.BloodProduct.GetBloodProductByGuid(this.BloodProductGuid);

					if (dt.Rows.Count == 0)
						return(false);
					else
						this._productAttributeForm = dt.Rows[0][BLOODPRODUCT.ProductAttributeForm].ToString();
				}

				return(this._productAttributeForm.IndexOf("B2") >= 0);
			}
		}


		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/3/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="347"> 
		///		<ExpectedInput>ProductShippers object</ExpectedInput>
		///		<ExpectedOutput>ProductShippers object</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="681"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets or sets a table containing Shippers of this blood product.
		/// </summary>
		public BOL.ProductShipperCollection ProductShippers
		{
			get
			{
				return this._productShippers;
			}
			set
			{
				this._productShippers = value;
				this.IsDirty = true;
			}
		}

		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/23/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1472"> 
		///		<ExpectedInput>Valid donation type</ExpectedInput>
		///		<ExpectedOutput>Valid donation type</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="574"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///	
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Donation type
		/// </summary>
		public DonationType DonationType
		{
			get
			{
				if (_donationType == null)
				{
					_donationType = new BOL.DonationType('0');
				}
				return _donationType;
			}
			set
			{
				_donationType = value;
				this.IsDirty = true;
			}
		}

		///<Developers>
		///	<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/21/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3357"> 
		///		<ExpectedInput>Valid string</ExpectedInput>
		///		<ExpectedOutput>Valid UnitDivisionId</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3358"> 
		///		<ExpectedInput>Invalid unit division ID length > 2</ExpectedInput>
		///		<ExpectedOutput>ArumentException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Unit Division ID
		/// </summary>
		public string UnitDivisionId
		{
			get
			{
				return _unitDivisionID;
			}
			set
			{
				if (value.Length > 2)
				{
					throw new ArgumentException(MethodBase.GetCurrentMethod().Name);
				}
				_unitDivisionID = value;
				this.IsDirty = true;
			}

		}

		#endregion

		#region Methods

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>6/21/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="575"> 
		///		<ExpectedInput>localSupplierGuid</ExpectedInput>
		///		<ExpectedOutput>boolean</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="576"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Checks the database for a real-time "is active" indicator.
		/// Removed from ProductShippers collection since the indicator could be updated by one user and a user holding
		/// the collection wouldn't get the change...
		/// </summary>
		/// <param name="localSupplierGuid"></param>
		/// <returns></returns>
		public bool IsProductShipperActive(System.Guid localSupplierGuid)
		{
			return (DAL.BloodProduct.IsProductShipperActive(this.BloodProductGuid, localSupplierGuid));
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/3/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="348"> 
		///		<ExpectedInput>Valid product name, product type code, and division code</ExpectedInput>
		///		<ExpectedOutput>DataTable containing blood products</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2427"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Static method to search database and return all blood products matching given product type and/or product name 
		/// (within given division only).
		/// </summary>
		/// <param name="productName"></param>
		/// <param name="productTypeCode"></param>
		/// <returns></returns>
		public static DataTable BloodProductSearch(string productName, string productTypeCode)
		{
			return DAL.BloodProduct.BloodProductSearch(productName, productTypeCode);
		}

		///<Developers>
		///	<Developer>Unknown</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>?/?/200?</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="577"> 
		///		<ExpectedInput>Valid bloodproductguid</ExpectedInput>
		///		<ExpectedOutput>bloodproduct object</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="578"> 
		///		<ExpectedInput>New Guid</ExpectedInput>
		///		<ExpectedOutput>New BloodProduct</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Populates the BloodProductObject.  Why not a constructor?  Because of the BloodUnit's need for this I guess. Architects???
		/// </summary>
		/// <param name="bloodProductGuid"></param>
		public void GetBloodProductByGuid(System.Guid bloodProductGuid)
		{
			//Grab our bloodproduct information
			DataTable dt = DAL.BloodProduct.GetBloodProductByGuid(bloodProductGuid);

			switch(dt.Rows.Count)
			{
				case 0:
					this._drBloodProduct = DAL.BloodProduct.GetEmptyBloodProductTable().NewRow();
					this.BloodProductGuid = bloodProductGuid;
					this.ProductShippers = new BOL.ProductShipperCollection(bloodProductGuid);
					this.ProductShippers.ListChanged += new System.EventHandler(OnProductShippersChanged);
					this.IsNew = true;
					this.IsDirty = true;
					break;
				case 1:
					this.LoadFromDataRow(dt.Rows[0]);
					this.IsNew = false;
					break;
				default:
					throw new TooManyRecordsFoundException(Common.StrRes.SysErrMsg.Common.NoRecordsFoundException("Blood product").ResString);
			}
		}

		///<Developers>
		///	<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/21/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3359"> 
		///		<ExpectedInput>Valid blood product guid</ExpectedInput>
		///		<ExpectedOutput>Data table with records</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3360"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// GetHcpcsAssignmentHistory
		/// </summary>
		/// <param name="bloodProductGuid"></param>
		/// <returns></returns>
		public static DataTable GetHcpcsAssignmentHistory(System.Guid bloodProductGuid)
		{
			return DAL.BloodProduct.GetHcpcsAssignmentHistory(bloodProductGuid);
		}

		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/11/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="509"> 
		///		<ExpectedInput>Valid DataRow</ExpectedInput>
		///		<ExpectedOutput>Loaded DataRow</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2453"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Public function to return a data row of the given data table schema, populated by blood property obejct properties
		/// </summary>
		/// <param name="dtRow"></param>
		/// <returns></returns>
		public new System.Data.DataRow LoadDataRowFromThis(System.Data.DataRow dtRow)
		{
			if (dtRow.Table.Columns.Contains(BLOODPRODUCT.BloodProductGuid))
			{
				dtRow[BLOODPRODUCT.BloodProductGuid] = this.BloodProductGuid;
			}

			if (dtRow.Table.Columns.Contains(BLOODPRODUCT.ProductCode))
			{
				dtRow[BLOODPRODUCT.ProductCode] = this.ProductCode;
			}

			if (dtRow.Table.Columns.Contains(BLOODPRODUCT.ProductShortName))
			{
				dtRow[BLOODPRODUCT.ProductShortName] = this.ProductShortName;
			}

			if (dtRow.Table.Columns.Contains(BLOODPRODUCT.ProductName))
			{
				dtRow[BLOODPRODUCT.ProductName] = this.ProductName;
			}

			if (dtRow.Table.Columns.Contains(BLOODPRODUCT.ProductTypeCode))
			{
				dtRow[BLOODPRODUCT.ProductTypeCode] = this.ProductType.ProductTypeCode;
			}

			if (dtRow.Table.Columns.Contains(PRODUCTTYPE.ProductTypeName))
			{
				dtRow[PRODUCTTYPE.ProductTypeName] = this.ProductType.ProductTypeText;
			}

            //CR3558 - populating new ProductType.AboRhConfirmationRequired column
            if (dtRow.Table.Columns.Contains(PRODUCTTYPE.AboRhConfirmationRequired))
            {
                dtRow[PRODUCTTYPE.AboRhConfirmationRequired] = this.ProductType.AboRhConfirmationRequired;
            }

			if (dtRow.Table.Columns.Contains(BLOODPRODUCT.ProductVolume))
			{
				dtRow[BLOODPRODUCT.ProductVolume] = this.ProductVolume;
			}

			if (dtRow.Table.Columns.Contains(BLOODPRODUCT.MaximumStorageTime))
				// Rule of thumb for calculated values, should store and retrieve exactly what was entered as well.
			{
				dtRow[BLOODPRODUCT.MaximumStorageTime] = (decimal)((this.MaximumStorageDays * 24) + (this.MaximumStorageHours));
			}

			if (dtRow.Table.Columns.Contains(ARTIFICIAL.ProductActiveIndicator))
			{
				dtRow[ARTIFICIAL.ProductActiveIndicator] = this.ProductActiveIndicator;
			}

			if (dtRow.Table.Columns.Contains(HCPCSASSIGNMENT.HcpcsCode))
			{
				dtRow[HCPCSASSIGNMENT.HcpcsCode] = this.HcpcsCode;
			}

			if (dtRow.Table.Columns.Contains(ARTIFICIAL.BarcodeType))
			{
				dtRow[ARTIFICIAL.BarcodeType] = (this.BarcodeType == BarcodeType.ISBT128);
			}

			return dtRow;
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/3/2002</CreationDate>
		/// <summary>
		/// protected method to populate a blood product object with values from given blood product data row.
		/// </summary>
		/// <param name="dtRow"></param>
		protected new void LoadFromDataRow(System.Data.DataRow dtRow)
		{
			if (dtRow.Table.Columns.Contains(BLOODPRODUCT.BloodProductGuid))
			{
				this.BloodProductGuid = (System.Guid)dtRow[BLOODPRODUCT.BloodProductGuid];
			}

			if (dtRow.Table.Columns.Contains(BLOODPRODUCT.ProductCode))
			{
				this.ProductCode = dtRow[BLOODPRODUCT.ProductCode].ToString();
			}

			if (dtRow.Table.Columns.Contains(BLOODPRODUCT.ProductShortName))
			{
				this.ProductShortName = dtRow[BLOODPRODUCT.ProductShortName].ToString();
			}

			if (dtRow.Table.Columns.Contains(BLOODPRODUCT.ProductName))
			{
				this.ProductName = dtRow[BLOODPRODUCT.ProductName].ToString();
			}

			if (dtRow.Table.Columns.Contains(BLOODPRODUCT.ProductTypeCode))
			{
				this.ProductType.ProductTypeCode = dtRow[BLOODPRODUCT.ProductTypeCode].ToString();
			}

			if (dtRow.Table.Columns.Contains(PRODUCTTYPE.ProductTypeName))
			{
				this.ProductType.ProductTypeText = dtRow[PRODUCTTYPE.ProductTypeName].ToString();
			}

            //CR3558 - populating the new property in ProductType
            if (dtRow.Table.Columns.Contains(PRODUCTTYPE.AboRhConfirmationRequired))
	        {
		        this.ProductType.AboRhConfirmationRequired = dtRow.IsNull(PRODUCTTYPE.AboRhConfirmationRequired) ? false : (bool)dtRow[PRODUCTTYPE.AboRhConfirmationRequired];
	        }

			if (dtRow.Table.Columns.Contains(BLOODPRODUCT.ProductVolume))
			{
				this.ProductVolume = dtRow.IsNull(BLOODPRODUCT.ProductVolume) ? (System.Int16)0 : (System.Int16)dtRow[BLOODPRODUCT.ProductVolume];
			}

			if (dtRow.Table.Columns.Contains(BLOODPRODUCT.MaximumStorageTime))
			{
				this.MaximumStorageDays = dtRow.IsNull(BLOODPRODUCT.MaximumStorageTime) ? 0 : (int)(((decimal)dtRow[BLOODPRODUCT.MaximumStorageTime]) / 24);
				this.MaximumStorageHours = dtRow.IsNull(BLOODPRODUCT.MaximumStorageTime) ? 0 : (int)(((decimal)dtRow[BLOODPRODUCT.MaximumStorageTime]) % 24);
			}

			if (dtRow.Table.Columns.Contains(HCPCSASSIGNMENT.HcpcsCode))
			{
				this.HcpcsCode = dtRow.IsNull(HCPCSASSIGNMENT.HcpcsCode) ? "" : dtRow[HCPCSASSIGNMENT.HcpcsCode].ToString();
			}

			if (dtRow.Table.Columns.Contains(Common.VbecsTables.ComponentClass.ComponentClassId))
			{
				this.BloodComponentClassId = dtRow.IsNull(Common.VbecsTables.ComponentClass.ComponentClassId) ? 0 : (System.Int32)dtRow[Common.VbecsTables.ComponentClass.ComponentClassId];
			}

			if (dtRow.Table.Columns.Contains(COMPONENTCLASS.ComponentClassName))
			{
				this.BloodComponentName = dtRow.IsNull(COMPONENTCLASS.ComponentClassName) ? string.Empty : dtRow[COMPONENTCLASS.ComponentClassName].ToString();
			}

			if (dtRow.Table.Columns.Contains(BLOODPRODUCT.ProductAttributeForm))
			{
				this.ProductAttributeForm = dtRow.IsNull(BLOODPRODUCT.ProductAttributeForm) ? string.Empty : (System.String)dtRow[BLOODPRODUCT.ProductAttributeForm];
			}
			if (dtRow.IsNull(ARTIFICIAL.BarcodeType))
			{
				this.BarcodeType = BarcodeType.None;
			}
			else
			{
				if ((bool)dtRow[ARTIFICIAL.BarcodeType])
				{
					this.BarcodeType = BarcodeType.ISBT128;
				}
				else
				{
					this.BarcodeType = BarcodeType.Codabar;
				}
			}

			this._productShippers = new BOL.ProductShipperCollection(this.BloodProductGuid);
			this._productShippers.ListChanged += new System.EventHandler(OnProductShippersChanged);

		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>6/3/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="579"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>Update ProductShippers object</ExpectedOutput>
		///</Case>
		///
		///<Case type="1" testid ="580"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Refreshes the list of product Shippers from the database for the current bloodproduct
		/// </summary>
		public void RefreshProductShippers()
		{
			this._productShippers = null;

			this._productShippers = new BOL.ProductShipperCollection(this.BloodProductGuid);
			this._productShippers.ListChanged += new System.EventHandler(OnProductShippersChanged);
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/11/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="510"> 
		///		<ExpectedInput>Valid product</ExpectedInput>
		///		<ExpectedOutput>DataTable containing blood products</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2434"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Public static function to return a datatable of blood products matching 
		/// a given product code within a particular division
		/// </summary>
		/// <param name="productCode"></param>
		/// <returns></returns>
		public static DataTable GetBloodProductsByCode(string productCode)
		{
			return DAL.BloodProduct.GetBloodProductByCode(productCode);
		}

		///<Developers>
		///	<Developer>D. Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>4/10/2007</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8479"> 
		///		<ExpectedInput>Valid ProductCode</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8480"> 
		///		<ExpectedInput>Invalid or Inactive ProductCode</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Public static function to return a datatable of blood products matching 
		/// a given product code within a particular division
		/// This version searches for both active and local roducts (CR 2210)	
		/// </summary>
		/// <param name="productCode"></param>
		/// <returns></returns>
		public static DataTable GetActiveOrLocalBloodProductsByCode(string productCode)
		{
			return DAL.BloodProduct.GetActiveOrLocalBloodProductsByCode(productCode);
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/3/2002</CreationDate>
		/// <summary>
		/// Private method to set the initial values for all properties of a blood product object.
		/// </summary>
		private new void Clear()
		{
			this.ProductType = new BOL.ProductType();
			this.BloodProductGuid = Guid.Empty;
			this.ProductCode = string.Empty;
			this.ProductShortName = string.Empty;
			this.ProductName = string.Empty;
			this.ProductVolume = 0;
			this.MaximumStorageDays = 0;
			this.MaximumStorageHours = 0;
			this.BarcodeType = BarcodeType.None;
		}

		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/26/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2451"> 
		///		<ExpectedInput>Valid product code and division code</ExpectedInput>
		///		<ExpectedOutput>true</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2452"> 
		///		<ExpectedInput>Empty string</ExpectedInput>
		///		<ExpectedOutput>false</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Checks if the blood product is active for the division
		/// Used to implement BR_26.11.
		/// </summary>
		/// <param name="bloodProductGuid"></param>
		/// <returns></returns>
		public static bool IsBloodProductActive(System.Guid bloodProductGuid)
		{
			return DAL.BloodProduct.IsBloodProductActive(bloodProductGuid);
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/3/2002</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="353"> 
		///		<ExpectedInput>Valid parameters</ExpectedInput>
		///		<ExpectedOutput>true</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2454"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		///<param name="lastUpdateFunctionId">UC or calling method</param>
		/// <summary>
		/// Public method to insert or update blood product within given division code.
		/// </summary>
		public void Save(Common.UpdateFunction lastUpdateFunctionId)
		{
			System.Collections.ArrayList dtArrayList = new System.Collections.ArrayList();
			System.Collections.ArrayList sprocArrayList = new System.Collections.ArrayList();

			bool updateHcpcs = false;
			bool insertHcpcs = true;

			if (this.HcpcsCode.Trim().Length > 0)
			{
				//Find out if we need to run the update first - by checking for existing history
				DataTable dtExistingHcpcs = DAL.BloodProduct.GetHcpcsAssignmentHistory(this.BloodProductGuid);

				if (dtExistingHcpcs.Rows.Count > 0)
				{
					updateHcpcs = true;
					if (dtExistingHcpcs.Rows[0][HCPCSASSIGNMENT.HcpcsCode].ToString().Trim() == this.HcpcsCode.Trim())
					{
						insertHcpcs = false;
						updateHcpcs = false;
					}
				}

				if (updateHcpcs)
				{
					DataTable dtUpdate = DAL.BloodProduct.GetEmptyHcpcsAssignmentTableSchema(true);
					System.Data.DataRow drUpdate = dtUpdate.NewRow();

					//We use the top row, because GetHcpcsAssignmentHistory returns rows in desc order 
					drUpdate[HCPCSASSIGNMENT.HcpcsAssignmentGuid] = dtExistingHcpcs.Rows[0][HCPCSASSIGNMENT.HcpcsAssignmentGuid];
					drUpdate[HCPCSASSIGNMENT.RowVersion] = dtExistingHcpcs.Rows[0][HCPCSASSIGNMENT.RowVersion];
					drUpdate[HCPCSASSIGNMENT.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
					drUpdate[HCPCSASSIGNMENT.LastUpdateFunctionId] = lastUpdateFunctionId;
					dtUpdate.Rows.Add(drUpdate);

					dtArrayList.Add(dtUpdate);
					sprocArrayList.Add(Common.VbecsStoredProcs.UpdateHcpcsAssignment.StoredProcName);
				}

				if (insertHcpcs)
				{
					//Create the table needed for the Insert
					DataTable dtInsert = DAL.BloodProduct.GetEmptyHcpcsAssignmentTableSchema(false);

					DataRow drInsert = dtInsert.NewRow();
					drInsert[HCPCSASSIGNMENT.BloodProductGuid] = this.BloodProductGuid;
					drInsert[HCPCSASSIGNMENT.HcpcsAssignmentGuid] = System.Guid.NewGuid();
					drInsert[HCPCSASSIGNMENT.HcpcsCode] = this.HcpcsCode;
					drInsert[HCPCSASSIGNMENT.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
					drInsert[HCPCSASSIGNMENT.LastUpdateFunctionId] = lastUpdateFunctionId;
					dtInsert.Rows.Add(drInsert);

					dtArrayList.Add(dtInsert);
					sprocArrayList.Add(Common.VbecsStoredProcs.InsertHcpcsAssignment.StoredProcName);
				}
			}

			DataTable dtInsPS = DAL.ProductShipper.GetEmptyProductShipperTable(false);
			DataTable dtUpdPS = DAL.ProductShipper.GetEmptyProductShipperTable(true);

			foreach(BOL.ProductShipper psObj in this.ProductShippers)
			{
				if (psObj.IsDirty)
				{
					if (psObj.IsNew)
					{
						dtInsPS.Rows.Add(psObj.LoadDataRowFromThis(dtInsPS.NewRow()));
					}
					else
					{
						dtUpdPS.Rows.Add(psObj.LoadDataRowFromThis(dtUpdPS.NewRow()));
					}
				}
			}

			dtArrayList.Add(Common.Utility.AppendLastUpdateInformation(dtInsPS, lastUpdateFunctionId));
			sprocArrayList.Add(Common.VbecsStoredProcs.InsertProductShipper.StoredProcName);

			dtArrayList.Add(Common.Utility.AppendLastUpdateInformation(dtUpdPS, lastUpdateFunctionId));
			sprocArrayList.Add(Common.VbecsStoredProcs.UpdateProductShipper.StoredProcName);

			int uselessVariable = new Common.StoredProcedure().TransactionalGetValue(sprocArrayList, dtArrayList);
		}


		///<Developers>
		///	<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/26/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2437"> 
		///		<ExpectedInput>Valid product Shipper GUID</ExpectedInput>
		///		<ExpectedOutput>product cost</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="2438"> 
		///		<ExpectedInput>Invalid local supplier GUID</ExpectedInput>
		///		<ExpectedOutput>NoRecordsFoundException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets the blood product's cost 
		/// </summary>
		/// <param name="localSupplierGuid"></param>
		/// <returns></returns>
		public decimal GetProductCost(System.Guid localSupplierGuid)
		{
			BOL.ProductShipper _ps = this.ProductShippers.GetElementByLocalSupplierGuid(localSupplierGuid);

			if (_ps == null)
			{
				string errorMessage = Common.StrRes.SysErrMsg.Common.NoRecordsFoundException("Product cost").ResString;
				throw new NoRecordsFoundException(errorMessage);
			}
			else
			{
				return _ps.ProductCost;
			}
		}

		///<Developers>
		///	<Developer>Mike      </Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>4/4/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2596"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>Valid data table</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="592"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Returns a data table of Hcpcs codes
		/// </summary>
		/// <returns></returns>
		public static DataTable GetHcpcsCodes()
		{
			if (_dtHcpcsCodes == null)
			{
				_dtHcpcsCodes = DAL.VAL.HcpcsCodes.GetHcpcsCodes();
			}

			return(_dtHcpcsCodes);
		}

		#endregion
	}
}